#!/bin/bash

###########################################################################
#
# mkvport.sh
# This script is provided by Emulex to use with its 8.2.x Linux device 
# drivers for Light Pulse Fibre Channel adapters.
#
# This script facilitates the creation of Virtual Ports (vports).
#
# Usage: 
#        mkvport.sh <scsi_host_name> <wwpn> <wwnn>
#
###########################################################################

enumerate_hosts() {
    hosts=$(ls -1 -d /sys/class/*_host/host*/* | grep vport_create | sed -e 's:/vport_create::')
    if [ "$hosts" = "" ]; then
	echo $1 there do not appear to be any Emulex Lightpulse HBA adapters capable
	echo of creating virtual ports on your system.  Perhaps the NPIV verion of
	echo the LPFC driver is not loaded.
    else
    	echo Usage: $0 "<scsi_host_name> <wwpn> <wwnn>"
	echo
	echo The following hosts represent Emulex Lightpulse HBA adapters on your system\:
	echo
	for i in $hosts; do
        host=$(echo $i | sed -e 's:/sys/class/scsi_host/::; s:/sys/class/fc_host/::')
        max=$(cat $i/max_npiv_vports)
        inuse=$(cat $i/npiv_vports_inuse)
	    echo "    $host (Max Vports:$max Vports In use:$inuse)"
	done
	echo
	echo So, for example,
	echo
	echo "    " $0 $host 10000000c94ac63a 20010000c94ac63a
	echo
	echo would create a virtual port with the World Wide Port Name 10000000c94ac63a
	echo and the World Wide Node Name 20010000c94ac63a
	echo
    fi
}

is_npiv_host() {
    info=$(cat /sys/class/scsi_host/${1}/npiv_info /dev/null 2>&1)
    if [ -z "${info}" ] ; then 
	return 0
    else
    info2=$(echo $info /dev/null 2>&1 |grep "NPIV Physical" )
	if [ $? -eq 0 ] ; then
	    return 1
	else
	    return 0
	fi
    fi
}

is_max_vport_limit_reached() {
    max_vports=$(cat "${vport_path}/max_npiv_vports")
    max_vports_used=$(cat "${vport_path}/npiv_vports_inuse")

    if [ "${max_vports}" -eq  "${max_vports_used}" ] ; then
	return 1
    else
	return 0
    fi
}

set_exit_state() {
    error_code=$3
    if [ "${error_code}" -ne 0 ] ; then
	state_file="/sys/class/${2}/${1}/state"
        if [ -e "${state_file}" ] ; then
	    cat $state_file
	fi
	exit 1
    else
	exit 0
    fi
}

#
# Program starts here
#

if [ $# -ne 3 ]; then
    enumerate_hosts
    exit 1
fi

# Fix the input to remove unwanted chars 
wwpn=$(echo $2|sed s";:;;g") 
wwnn=$(echo $3|sed s";:;;g")

if [ -e /sys/class/scsi_host/$1/vport_create ]; then
    is_npiv_host $1
    status=$?

    if [ ${status} -eq 0 ] || [ ! -w /sys/class/scsi_host/$1/vport_create ]; then
        echo $1 is not an Emulex HBA capable of supporting NPIV
	echo
        enumerate_hosts "In fact,"
        exit 1
    fi
    vport_path="/sys/class/scsi_host/$1"
    is_max_vport_limit_reached
    status=$?
    if [ ${status} -eq 0 ] ; then
	echo $wwpn:$wwnn > /sys/class/scsi_host/$1/vport_create
	set_exit_state $1 "scsi_host" $?
    else
	cat<<EOF
    Maximum vports limit exceeded. Maximum allowed vports : $max_vports
EOF
	exit 1
    fi
fi

if [ -e /sys/class/fc_host/$1/vport_create ]; then
    is_npiv_host $1
    status=$?
    if [ ${status} -eq 0 ] || [ ! -w /sys/class/fc_host/$1/vport_create ]; then
        echo $1 is not an Emulex HBA capable of supporting NPIV
        enumerate_hosts "In fact,"
        exit 1
    fi
    vport_path="/sys/class/fc_host/$1"
    is_max_vport_limit_reached
    status=$?
    if [ ${status} -eq 0 ] ; then
	echo $wwpn:$wwnn > /sys/class/fc_host/$1/vport_create
	set_exit_state $1 "fc_host" $?
    else
	cat<<EOF
    Maximum vports limit exceeded. Maximum allowed vports : $max_vports
EOF
	exit 1
    fi
fi

echo $1 is not an Emulex HBA capable of supporting NPIV
enumerate_hosts "In fact,"
exit 1
