#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Python structures for the Migration v2 stream format.
# See docs/specs/libxc-migration-stream.pandoc

# Image Header
IHDR_FORMAT = "!QIIHHI"

IHDR_MARKER  = 0xffffffffffffffff
IHDR_IDENT   = 0x58454E46 # "XENF" in ASCII
IHDR_VERSION = 2

IHDR_OPT_ENDIAN_ = 0
IHDR_OPT_LE = (0 << IHDR_OPT_ENDIAN_)
IHDR_OPT_BE = (1 << IHDR_OPT_ENDIAN_)

IHDR_OPT_RESZ_MASK = 0xfffe

# Domain Header
DHDR_FORMAT = "IHHII"

DHDR_TYPE_x86_pv  = 0x00000001
DHDR_TYPE_x86_hvm = 0x00000002
DHDR_TYPE_x86_pvh = 0x00000003
DHDR_TYPE_arm     = 0x00000004

dhdr_type_to_str = {
    DHDR_TYPE_x86_pv  : "x86 PV",
    DHDR_TYPE_x86_hvm : "x86 HVM",
    DHDR_TYPE_x86_pvh : "x86 PVH",
    DHDR_TYPE_arm     : "ARM",
}

RH_FORMAT = "II"

REC_TYPE_end                  = 0x00000000
REC_TYPE_page_data            = 0x00000001
REC_TYPE_x86_pv_info          = 0x00000002
REC_TYPE_x86_pv_p2m_frames    = 0x00000003
REC_TYPE_x86_pv_vcpu_basic    = 0x00000004
REC_TYPE_x86_pv_vcpu_extended = 0x00000005
REC_TYPE_x86_pv_vcpu_xsave    = 0x00000006
REC_TYPE_shared_info          = 0x00000007
REC_TYPE_tsc_info             = 0x00000008
REC_TYPE_hvm_context          = 0x00000009
REC_TYPE_hvm_params           = 0x0000000a
REC_TYPE_toolstack            = 0x0000000b
REC_TYPE_x86_pv_vcpu_msrs     = 0x0000000c
REC_TYPE_verify               = 0x0000000d

rec_type_to_str = {
    REC_TYPE_end                  : "End",
    REC_TYPE_page_data            : "Page data",
    REC_TYPE_x86_pv_info          : "x86 PV info",
    REC_TYPE_x86_pv_p2m_frames    : "x86 PV P2M frames",
    REC_TYPE_x86_pv_vcpu_basic    : "x86 PV vcpu basic",
    REC_TYPE_x86_pv_vcpu_extended : "x86 PV vcpu extended",
    REC_TYPE_x86_pv_vcpu_xsave    : "x86 PV vcpu xsave",
    REC_TYPE_shared_info          : "Shared info",
    REC_TYPE_tsc_info             : "TSC info",
    REC_TYPE_hvm_context          : "HVM context",
    REC_TYPE_hvm_params           : "HVM params",
    REC_TYPE_toolstack            : "Toolstack",
    REC_TYPE_x86_pv_vcpu_msrs     : "x86 PV vcpu msrs",
    REC_TYPE_verify               : "Verify",
}

# page_data
PAGE_DATA_FORMAT             = "II"
PAGE_DATA_PFN_MASK           = (1L << 52) - 1
PAGE_DATA_PFN_RESZ_MASK      = ((1L << 60) - 1) & ~((1L << 52) - 1)

# flags from xen/public/domctl.h: XEN_DOMCTL_PFINFO_* shifted by 32 bits
PAGE_DATA_TYPE_SHIFT         = 60
PAGE_DATA_TYPE_LTABTYPE_MASK = (0x7L << PAGE_DATA_TYPE_SHIFT)
PAGE_DATA_TYPE_LTAB_MASK     = (0xfL << PAGE_DATA_TYPE_SHIFT)
PAGE_DATA_TYPE_LPINTAB       = (0x8L << PAGE_DATA_TYPE_SHIFT) # Pinned pagetable

PAGE_DATA_TYPE_NOTAB         = (0x0L << PAGE_DATA_TYPE_SHIFT) # Regular page
PAGE_DATA_TYPE_L1TAB         = (0x1L << PAGE_DATA_TYPE_SHIFT) # L1 pagetable
PAGE_DATA_TYPE_L2TAB         = (0x2L << PAGE_DATA_TYPE_SHIFT) # L2 pagetable
PAGE_DATA_TYPE_L3TAB         = (0x3L << PAGE_DATA_TYPE_SHIFT) # L3 pagetable
PAGE_DATA_TYPE_L4TAB         = (0x4L << PAGE_DATA_TYPE_SHIFT) # L4 pagetable
PAGE_DATA_TYPE_BROKEN        = (0xdL << PAGE_DATA_TYPE_SHIFT) # Broken
PAGE_DATA_TYPE_XALLOC        = (0xeL << PAGE_DATA_TYPE_SHIFT) # Allocate-only
PAGE_DATA_TYPE_XTAB          = (0xfL << PAGE_DATA_TYPE_SHIFT) # Invalid

# x86_pv_info
X86_PV_INFO_FORMAT        = "BBHI"

X86_PV_P2M_FRAMES_FORMAT  = "II"

# x86_pv_vcpu_{basic,extended,xsave,msrs}
X86_PV_VCPU_HDR_FORMAT    = "II"

# tsc_info
TSC_INFO_FORMAT           = "IIQII"

# hvm_params
HVM_PARAMS_ENTRY_FORMAT   = "QQ"
HVM_PARAMS_FORMAT         = "II"

#
# libxl format
#

LIBXL_QEMU_SIGNATURE = "DeviceModelRecord0002"
LIBXL_QEMU_RECORD_HDR = "=%dsI" % (len(LIBXL_QEMU_SIGNATURE), )


def __main():
    """ Basic sanity check for format sizes """

    import sys
    from struct import calcsize

    ok = True
    for fmt, sz in [ ("IHDR_FORMAT", 24),
                     ("DHDR_FORMAT", 16),
                     ("RH_FORMAT", 8),

                     ("PAGE_DATA_FORMAT", 8),
                     ("X86_PV_INFO_FORMAT", 8),
                     ("X86_PV_P2M_FRAMES_FORMAT", 8),
                     ("X86_PV_VCPU_HDR_FORMAT", 8),
                     ("TSC_INFO_FORMAT", 24),
                     ("HVM_PARAMS_ENTRY_FORMAT", 16),
                     ("HVM_PARAMS_FORMAT", 8),
                     ]:

        realsz = calcsize(getattr(sys.modules[__name__], fmt))
        if realsz != sz:
            print "%s is %d bytes but expected %d" % (fmt, realsz, sz)
            ok = False

    if ok:
        sys.exit(0)
    else:
        sys.exit(1)

# If run as a python script alone, confirm some expected sizes
if __name__ == "__main__":
    __main()
