#!/usr/bin/env python

from xcp.bootloader import *
from xcp.environ import *
from optparse import OptionParser
import glob
import os
import os.path
import re
import subprocess
import sys

def kernel_install(version, options):
    kernel_name = os.path.join('/boot', 'vmlinuz-'+version)
    initrd_name = os.path.join('/boot', 'initrd-'+version+'.img')
    suffix = ''
    if options.package and options.package.startswith('kernel-'):
        suffix = options.package[6:]
    m = re.match(r'\d+\.\d+', version)
    if not m:
        raise SystemExit, "Invalid version format"
    kernel_link_name = os.path.join('/boot', 'vmlinuz-%s%s' % (m.group(0), suffix))
    initrd_link_name = os.path.join('/boot', 'initrd-%s%s.img' % (m.group(0), suffix))

    if not os.path.exists(kernel_name):
        raise SystemExit, "Cannot locate kernel: "+kernel_name

    # create module dependencies
    if options.moddep == 'add':
        if options.verbose:
            print "Building module dependencies for "+version
        if subprocess.call(['depmod', '-ae', '-F', '/boot/System.map-'+version, version]) != 0:
            raise SystemExit, "Failed to create module dependencies"

    if options.initrd == 'add':
        # create initrd and symlink
        args = ['-f']
        if options.verbose:
            print "Building initrd for "+version
            args.append('-v')
        if subprocess.call(['mkinitrd'] + args + [initrd_name, version]) != 0:
            raise SystemExit, "Failed to create initrd"

        if options.make_default:
            if options.verbose:
                print "Updating initrd symlink"
            if os.path.exists(initrd_link_name):
                os.unlink(initrd_link_name)
            os.symlink(os.path.basename(initrd_name), initrd_link_name)

    if options.make_default:
        # create kernel symlink
        if options.verbose:
            print "Updating kernel symlink"
        if os.path.exists(kernel_link_name):
            os.unlink(kernel_link_name)
        os.symlink(os.path.basename(kernel_name), kernel_link_name)

        if not installerRunning() and not buildingInitialTar():
            if options.verbose:
                print "Updating bootloader"
            try:
                Bootloader.newDefault(kernel_link_name, initrd_link_name)
            except RuntimeError, s:
                raise SystemExit, s

def kernel_remove(version, options):
    initrd_name = os.path.join('/boot', 'initrd-'+version+'.img')

    # remove module dependencies
    if options.moddep == 'del':
        if options.verbose:
            print "Removing module dependencies for "+version
        modfiles = glob.glob(os.path.join('/lib/modules', version, 'modules.*'))
        for f in modfiles:
            try:
                os.unlink(f)
            except:
                pass

    # remove initrd
    if options.initrd == 'del':
        if options.verbose:
            print "Removing initrd for "+version
        try:
            os.unlink(initrd_name)
        except:
            pass


if __name__ == '__main__':
    parser = OptionParser(usage="usage: %prog [options] kernel-version")
    parser.set_defaults(mode=None, initrd=None, moddep=None, make_default=False, verbose=False)
    parser.add_option('--install', action='store_const', const='install', dest='mode')
    parser.add_option('--remove', action='store_const', const='remove', dest='mode')
    parser.add_option('--mkinitrd', action='store_const', const='add', dest='initrd')
    parser.add_option('--rminitrd', action='store_const', const='del', dest='initrd')
    parser.add_option('--depmod', action='store_const', const='add', dest='moddep')
    parser.add_option('--rmmoddep', action='store_const', const='del', dest='moddep')
    parser.add_option('--make-default', action='store_true', dest='make_default')
    parser.add_option('--verbose', action='store_true', dest='verbose')
    parser.add_option('--package', dest='package')
    # for backwards compatibility
    parser.add_option('--multiboot', dest='multiboot')

    (options, args) = parser.parse_args()

    if options.mode not in ['install', 'remove']:
        parser.print_help()
        sys.exit(1)

    if (options.initrd == 'add' or options.moddep == 'add') and options.mode != 'install':
        parser.print_help()
        sys.exit(1)

    if (options.initrd == 'del' or options.moddep == 'del') and options.mode != 'remove':
        parser.print_help()
        sys.exit(1)

    if len(args) != 1:
        parser.print_help()
        sys.exit(1)

    version = args[0]

    if options.initrd == 'add' and buildingInitialTar():
        if options.verbose:
            print "In build environment, not creating initrd"
        options.initrd = None

    if options.mode == 'install':
        kernel_install(version, options)
    elif options.mode == 'remove':
        kernel_remove(version, options)

    sys.exit(0)
