#
# Copyright (c) Citrix Systems 2008. All rights reserved.
#

DISKPREP="/opt/xensource/bin/diskprep"

. "$XENSOURCE_INVENTORY"

TAG="storage-creation-utils"

##
# wait_for_sm_plugin type
#
# Waits for the given plugin to be registered, so we can create SRs with it.
wait_for_sm_plugin() {
    local type="$1"
    shift 1
    MAX_RETRIES=300
    RETRY=0
    logger -t "${TAG}" "Waiting for SM plugin ${type} to be registered"
    while [ ${RETRY} -lt ${MAX_RETRIES} ]; do
		# NB 10-prepare-storage runs with 'set -e'
		uuid=$($XE sm-list type=${type} params=uuid --minimal 2>/dev/null || true)
		if [ ! -z "${uuid}" ]; then
            logger -t "${TAG}" "detected SM plugin ${type} complete after ${RETRY} / ${MAX_RETRIES} s"
            return 0
        fi
        sleep 1
        echo -n "."
        RETRY=$(( ${RETRY} + 1 ))
    done
    logger -t "${TAG}" "failed to detect SM plugin ${type} after ${MAX_RETRIES}s"
    echo "failed to detect SM plugin ${type} after ${MAX_RETRIES}s"
    return 1
}


##
# sr_create name-label name-description type content-type i18n-key i18n-index
#           <all device configuration>
#
# Creates an SR, using the given values, on the current host (using
# INSTALLATION_UUID from xensource-inventory).
#
# i18n-key may be one of "local-storage", "removable-storage",
# "local-hotplug-disk", "local-hotplug-cd", or other keys as recognized by
# XenCenter in the future.
#
# i18n-index should be an integer, if the i18n-key will be used more than
# once, or the empty string if only one SR of that kind will be created.
#
sr_create()
{
    local name_label="$1"
    local name_description="$2"
    local type="$3"
    local content_type="$4"
    local i18n_key="$5"
    local i18n_index="$6"

    shift 6

    wait_for_sm_plugin ${type}

    local sr_uuid=$($XE sr-create name-label="$name_label" \
                                  type="$type" content-type="$content_type" \
                                  host-uuid="$INSTALLATION_UUID" "$@")
    if [ ! "$sr_uuid" ]
    then
        echo "SR creation failed." >&2
        exit 1
    fi
    $XE sr-param-set uuid="$sr_uuid" other-config:i18n-key="$i18n_key"
    $XE sr-param-set uuid="$sr_uuid" other-config:i18n-original-value-name_label="$name_label"

    if [ "$name_description" ]
    then
        $XE sr-param-set uuid="$sr_uuid" name-description="$name_description"
        $XE sr-param-set uuid="$sr_uuid" other-config:i18n-original-value-name_description="$name_description"
    fi

    if [ "$i18n_index" ]
    then
        $XE sr-param-set uuid="$sr_uuid" other-config:i18n-index="$i18n_index"
    fi
}


##
# mk_non_spanning_local_srs partitions type
#
# Create a number of SRs, one for each of the given partitions, using the
# given type.  If just one partition is supplied, then the SR will be called
# "Local storage", otherwise the SRs will be called "Local storage N" where
# N is an integer.  The internationalisation keys will be set accordingly.
#
# partitions should be an IFS-separated string of partition names.
#
mk_non_spanning_local_srs()
{
    local partitions="$1"
    local type="$2"

    local tmp_arr=($partitions)
    local use_suffix=$(expr ${#tmp_arr[*]} != "1" || true)
    local i=1
    for p in $partitions
    do
        local name="Local storage"
        local index=""
        if [ "$use_suffix" == "1" ]
        then
            name="$name $i"
            index="$i"
            i=$(( $i + 1 ))
        fi
        sr_create "$name" "" "$type" "user" "local-storage" "$index" \
            "device-config:device=$p"
    done
}


mk_ext_sr()
{
    local partitions="$1"

    for p in $partitions
    do
        $DISKPREP -f "$p"
    done

    local partitions_cs=$(echo "$partitions" | sed "s/ /,/g")
    sr_create "Local storage" "" "ext" "user" "local-storage" "" \
        "device-config:device=$partitions_cs"
}


##
# mk_lvm_sr partitions
#
# Create a single "Local storage" SR, with a single LVM volume spanning
# the given partitions.
#
# partitions should be an IFS-separated string of partition names.
#
mk_lvm_sr()
{
    local partitions="$1"

    for p in $partitions
    do
        $DISKPREP -f "$p"
    done

    local partitions_cs=$(echo "$partitions" | sed "s/ /,/g")
    sr_create "Local storage" "" "lvm" "user" "local-storage" "" \
        "device-config:device=$partitions_cs"
}
