#!/bin/bash

# enqueue-interface-rename.sh written by Andrew Cooper
#
# Designed to be called from udev (net-rename-sideways.sh, which is invoked by
# udev in /etc/udev/rules.d/60-net.rules) to call interface-rename.py if needed.
# This script is triggered by the PROGRAM clause of the 'catch all' rule for
# unknown ethernet devices.
#
# During startup, there are potentially many unknown devices, causing this
# script to be invoked many times. Therefore, we use a lock to prevent race
# conditions, and only actually invoke interface-rename.py if there are actually
# devices needing to be renamed.  One problem with startup is that the root file
# system is usually read only, causing a problem for interface-rename.py.  In
# this case, bail as we explicitly call interface-rename.py in rc.sysinit once
# we get a writable filesystem.
#
# In the non startup case, this script could be called if someone is modprobing
# new drivers, exposing new hardware, in which we really do need re-call
# interface-rename.py



# Due to some 'fun' reason which is not clear from the source, udev adds pipes
# into its PROGRAM/RUN tasks which get inherited by child processes.  As a
# result, even though net-rename-sideways.sh disowns us and exits (causing us to
# be orphaned off to a child of init), udev keeps a handle on the pipe so
# notices that we are still alive.  This is a problem because it causes a
# deadlock with udevsettle as udev waits for us before it can settle, while we
# are waiting for it to settle.  Therefore, close all our file descriptors to
# fool udev into thinking we are done.  Then udevsettle works as intended.
for fd in $(cd /proc/$$/fd/; echo *);
do
    if [[ "$fd" == "255" ]];   then continue; fi
    eval "exec $fd>&-"
    eval "exec $fd<&-"
done

if [ -n "$(touch /root 2>&1 | grep Read-only)" ];
then
    # We have a read only filesystem so interface-rename.py wont work.
    # It should be called by rc.sysinit when / is remounted as rw.
    exit 0
fi

# Wait for udev to settle because interface-rename.py needs a steady state
/sbin/udevsettle --timeout=20

# open lock file for reading, as the root filesystem may be read only
exec 8<"/etc/udev/scripts/enqueue-interface-rename.lock"
/usr/bin/flock -nb -x 8

if [[ $? -eq 0 ]];
    # If flock returned 0, it successfully took the lock
then
    # devs is empty for no sideways devs, or non-empty if sideways devs exist
    devs=$(shopt -s nullglob; echo /sys/class/net/side-*)

    if [ -n "$devs" ]
    then
        # If there are devices named sideways, call interface-rename.py
        /usr/bin/python /etc/sysconfig/network-scripts/interface-rename.py --rename
    fi
#else
    # If flock returned non 0, it failed to take the lock.  This means we are
    # racing with ourselfs, so we can safely do nothing, as the other racer
    # will invoke interface-rename.py
fi

# Close filehandle to be neat.  This releases the lock
exec 8<&-
